<?php

namespace App\Http\Controllers\Api\Auth;

use App\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\UsersResources;
use App\Models\Offer;
use App\Models\Package;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Mollie\Laravel\Facades\Mollie;
use PHPOpenSourceSaver\JWTAuth\Facades\JWTAuth;
use PHPOpenSourceSaver\JWTAuth\Exceptions\JWTException;
use App\Models\User;
use Validator;

class AuthController extends Controller
{
    use ApiResponse;
    public function login(Request $request)
    {
        $credentials = $request->only('email', 'password');
        try {
            $credentials['user_type'] = 'Client';
            if (!$token = JWTAuth::attempt($credentials)) {
                return $this->apiResponse(400, 'invalid', ['Invalid credentials'], null);
            }
        } catch (JWTException $e) {
            return $this->apiResponse(500, 'Server Errors', ['Server Errors'], null);
        }
        $users = Auth()->user();
        $my = new UsersResources($users);
        $data['token'] = 'bearer ' . $token;
        $data['data'] = $my;
        return $this->apiResponse(200, 'User Data Login', null, $data);
    }
    public function me()
    {
        if (Auth()->guard('api')->check()) {
            $user = Auth()->guard('api')->user();
            $my = new UsersResources($user);
            return $this->apiResponse(200, 'User Data', null, $my);
        }
        return $this->apiResponse(403, 'invalid', ['unauthentic'], null);
    }
    public function register(Request $request)
    {
        $data = $request->all();
        $rules = [
            'name' => 'required|max:150',
            'email' => 'required|email|max:150|unique:users,email',
            'password' => 'required|confirmed|max:150',
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $errors = $validator->errors();
            return $this->apiResponse(400, 'errors data', $errors, null);
        }

        // created users and make login and retrun payment view :)
        if ($user = $this->createdUser($data)) {
            $credentials['email'] = $data['email'];
            $credentials['password'] = $data['password'];
            if ($token = JWTAuth::attempt($credentials)) {
                $my = new UsersResources($user);
                $retrunUser['token'] = 'bearer ' . $token;
                $retrunUser['data'] = $my;

                return $this->apiResponse(200, 'data user', null, $retrunUser);
            }
        }
        return $this->apiResponse(400, 'errors data', ["can't save user or get payment"], null);
    }
    public function updated(Request $request)
    {
        $data = $request->all();
        $rules = [
            'name' => 'required|max:150',
            'email' => 'required|email|max:150|unique:users,email,' . Auth()->guard('api')->user()->id,
            'phone' => 'nullable|numeric|digits_between:11,50',
            'gender' => 'nullable|max:150',
            'enable_sound' => 'nullable',
            'question_auto_move' => 'nullable',
        ];

        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $errors = $validator->errors();
            return $this->apiResponse(400, 'errors data', $errors, null);
        }

        $user = User::find(Auth()->guard('api')->user()->id);
        $user->name = $data['name'];
        $user->email = $data['email'];
        $user->phone = $data['phone'];
        $user->gender = $data['gender'];
        $user->enabel_sound = $data['enable_sound'];
        $user->question_auto_move = $data['question_auto_move'];
        $user->save();

        return $this->apiResponse(200, 'updated successfully', null, null);
    }
    public function password(Request $request)
    {
        $data = $request->all();
        $rules = [
            'password' => 'required|confirmed|max:150',
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $errors = $validator->errors();
            return $this->apiResponse(400, 'errors data', $errors, null);
        }

        $user = User::find(Auth()->guard('api')->user()->id);
        $user->password = Hash::make($data['password']);
        $user->save();

        return $this->apiResponse(200, 'updated successfully', null, null);
    }
    private function createdUser($data)
    {
        $create = new User();
        $create->name = $data['name'];
        $create->user_type = 'Client';
        $create->email = $data['email'];
        $create->password = Hash::make($data['password']);
        $create->email_verified_at = date('Y-m-d H:i:s');
        $create->save();
        return $create;
    }
    private function payment($data, $user)
    {
        $package = Package::find($data['package']);
        if ($package->offer) {
            $type = 'offer';
            $id = $package->offer->id;
        } else {
            $type = 'package';
            $id = $package->id;
        }

        if ($type == 'package') {
            if ($package) {
                $payment = Mollie::api()->payments->create([
                    "amount" => [
                        "currency" => "EUR",
                        "value" => number_format($package->price, 2, '.', '')  // You must send the correct number of decimals, thus we enforce the use of strings
                    ],
                    "description" => "Purchase Package #" . $package->{'name_' . App::getLocale()},
                    "redirectUrl" => route('general.redirectUrl'),
                    "webhookUrl" => route('webhooks.mollie', "package"),
                    "metadata" => [
                        "order_id" => $id . $user->id,
                        "price" => $package->price,
                        "expiration_duration_in_dayes" => $package->expiration_duration_in_dayes,
                        "package_id" => $package->id,
                        "offer_discount" => 0,
                        "offer_id" => null,
                        "user_id" => $user->id,
                        "user_name" => $user->name,
                        "lang" => App::getLocale(),
                    ],
                    "method" => $data['payment'],
                ]);
                $url = $payment->getCheckoutUrl();
                return $url;
            }
        } elseif ($type == 'offer') {
            $offer = Offer::find($id);
            if ($offer) {
                if (($offer->package->price - $offer->discount_amount) < 0) {
                    // return redirect()->back()->with("error1", trans('messages.This offer is expired you can subscrib available offers'));
                }
                $payment = Mollie::api()->payments->create([
                    "amount" => [
                        "currency" => "EUR",
                        "value" => number_format(($offer->package->price - $offer->discount_amount), 2, '.', '')  // You must send the correct number of decimals, thus we enforce the use of strings
                    ],
                    "description" => "Purchase Package #" . $offer->package->{'name_' . App::getLocale()},
                    "redirectUrl" => route('general.redirectUrl'),
                    "webhookUrl" => route('webhooks.mollie', "package"),
                    "metadata" => [
                        "order_id" => $id . $user->id,
                        "price" => $offer->package->price,
                        "expiration_duration_in_dayes" => $offer->package->expiration_duration_in_dayes,
                        "package_id" => $offer->package->id,
                        "offer_discount" => $offer->discount_amount,
                        "offer_id" => $offer->id,
                        "user_id" => $user->id,
                        "user_name" => $user->name,
                        "lang" => App::getLocale(),
                    ],
                    "method" => $data['payment'],
                ]);
                $url = $payment->getCheckoutUrl();
                return $url;
            }
        }

        return null;
    }
    public function getResetPassword(Request $request)
    {
        $data = $request->all();
        $rules = [
            'email' => 'required|email|exists:users,email'
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $errors = $validator->errors();
            return $this->apiResponse(400, 'errors data', $errors, null);
        }

        $otp = rand(1000, 9999);

        DB::table('password_resets')->updateOrInsert(
            ['email' => $request->email],
            [
                'otp' => $otp,
                'token' => 'dfasdfsasad',
                'created_at' => Carbon::now()->addMinutes(5),
            ]
        );

        // send mail
        // Mail::to();
        return $this->apiResponse(200, 'sent otp', null, ['otp' => $otp]);
    }
    public function ressetpassword(Request $request)
    {
        $data = $request->all();
        $rules = [
            'email' => 'required|email|exists:users,email',
            'otp' => 'required|digits:4',
            'password' => 'required|min:6|confirmed'
        ];
        $validator = Validator::make($data, $rules);
        if ($validator->fails()) {
            $errors = $validator->errors();
            return $this->apiResponse(400, 'errors data', $errors, null);
        }
        $record = DB::table('password_resets')
            ->where('email', $request->email)
            ->where('otp', $request->otp)
            ->first();

        if (!$record || Carbon::now()->greaterThan($record->created_at)) {
            return $this->apiResponse(400, 'opt expired', ['Invalid or expired OTP'], null);
        }

        $user = User::where('email', $request->email)->first();
        $user->password = Hash::make($request->password);
        $user->save();

        DB::table('password_resets')->where('email', $request->email)->delete();
        return $this->apiResponse(200, 'Password reset successfully', null, null);
    }
}
