<?php

namespace App\Sys\Repositories\Exams;

use App\Models\Exam;
use App\Models\Result;
use App\Models\Package;
use App\Models\Setting;
use App\Models\Subscription;

class ExamRepository
{
    protected $exam;

    public function __construct(Exam $exam)
    {
        $this->exam = $exam;
    }

    public function getLatestSubscriptionDate($userId)
    {
        return Subscription::query()
            ->where('user_id', $userId)
            ->where('expiration_date', '>', now())
            ->latest('subscription_date')
            ->value('subscription_date');
    }

    public function getActivePackageIds($userId)
    {
        return Package::query()
            ->whereHas('subscriptions', function ($query) use ($userId) {
                $query->where('user_id', $userId)
                    ->where('expiration_date', '>', now());
            })
            ->pluck('id');
    }

    public function getExamsBySubscribedPackages($limit, $activePackageId)
    {
        return $this->exam->whereHas('packages', function ($query) use ($activePackageId) {
            $query->whereIn('package_id', $activePackageId);
        })
            ->paginate($limit);
    }

    public function getExamsBySubscribedPackagesWithResults($userId, $activePackageId, $latestSubscriptionDate)
    {
        return $this->exam->whereHas('packages', function ($query) use ($activePackageId) {
            $query->whereIn('package_id', $activePackageId);
        })->with(['results' => function ($query) use ($userId, $latestSubscriptionDate) {
            $query->where('user_id', $userId)
                ->where('created_at', '>=', $latestSubscriptionDate);
        }])->get();
    }

    public function getCurrentAttempts($userId, $latestSubscriptionDate)
    {
        return Result::query()
            ->where('user_id', $userId)
            ->where('created_at', '>=', $latestSubscriptionDate)
            ->select('exam_id')
            ->selectRaw('COUNT(*) as attempt_count')
            ->groupBy('exam_id')
            ->pluck('attempt_count', 'exam_id');
    }

    public function getAttemptsUsed($userId, $examId, $latestSubscriptionDate)
    {
        return Result::query()
            ->where('user_id', $userId)
            ->where('exam_id', $examId)
            ->where('created_at', '>=', $latestSubscriptionDate)
            ->count();
    }

    public function getExamWithCategoryAndQuestions($examId)
    {
        return Exam::with(['examCategory.questions.answers'])->find($examId);
    }

    public function isExamInSubscribedPackages($examId, $activePackageIds)
    {
        return Exam::whereHas('packages', function ($query) use ($activePackageIds) {
            $query->whereIn('package_id', $activePackageIds);
        })
            ->where('id', $examId)
            ->exists();
    }

    public function getTestExam()
    {
        return  Setting::select('test_exam_id')->find(1);
    }

    public function createResultRecord($userId, $exam, int $attemptsUsed)
    {
        $result = new Result();
        $result->user_id = $userId;
        $result->exam_id = $exam->id;
        $result->attempt_num = $attemptsUsed + 1; //Current attempt number - Just for ref Won't be used to count attempts.
        $result->score = 0.00;
        $result->passed_exam = 0;
        $result->json_score = null;
        $result->total_current_questions = $exam->questions_num;
        $result->total_right_questions = 0;
        $result->total_wrong_questions = 0;
        $result->total_skiped_questions = 0;
        $result->total_not_answered_questions = $exam->questions_num;
        $result->total_flaged_questions = 0;
        $result->save();

        return $result;
    }

    public function updateResultRecord($request)
    {
        $result = Result::findOrFail($request['result_record']);

        // Calculate the score (handle division by zero)
        $totalQuestions = $result->total_current_questions;
        $score = $totalQuestions > 0 ? ($request['total_right_questions'] * 100) / $totalQuestions : 0;

        // Update the result instance properties
        $result->score = $score;
        $result->passed_exam = $request['passed_exam'];
        $result->json_score = isset($request['json_score']) ? $request['json_score'] : null;
        $result->total_right_questions = $request['total_right_questions'];
        $result->total_wrong_questions = $request['total_wrong_questions'];
        $result->total_skiped_questions = $request['total_skiped_questions'];
        $result->total_not_answered_questions = $request['total_not_answered_questions'];

        $result->save();

        return $result;
    }
}
