<?php

namespace App\Sys\Services\Exams;

use App\Sys\Services;
use Illuminate\Support\Facades\Validator;
use App\Sys\Repositories\Exams\ExamRepository;

class ExamService extends Services
{
    protected $examRepository;

    public function __construct(ExamRepository $examRepository)
    {
        $this->examRepository = $examRepository;
    }

    public function getExamsForSubscribedPackages($userId, $limit)
    {

        // Get data from the repository
        $latestSubscriptionDate = $this->getLatestSubscriptionDate($userId);
        $activePackageIds = $this->getActivePackageIds($userId);

        // Return false if no active subscriptions or packages
        if (!$latestSubscriptionDate || $activePackageIds->isEmpty()) {
            return false;
        }

        $exams = $this->examRepository->getExamsBySubscribedPackages($limit, $activePackageIds);

        $currentAttempts = $this->examRepository->getCurrentAttempts($userId, $latestSubscriptionDate);

        return [
            'exams' => $exams,
            'current_attempts' => $currentAttempts,
        ];
    }

    public function getSubscribedPackageExamQuestions($userId, $examId)
    {
        $exam = $this->examRepository->getExamWithCategoryAndQuestions($examId);

        if (!$exam) {
            $this->setError('Exam not found');
            return false;
        }

        $activePackageIds = $this->getActivePackageIds($userId);

        if (!$this->examRepository->isExamInSubscribedPackages($examId, $activePackageIds)) {
            $this->setError('You are not subscribed to this exam\'s package');
            return false;
        }

        $latestSubscriptionDate = $this->getLatestSubscriptionDate($userId);

        $attemptsUsed = $this->examRepository->getAttemptsUsed($userId, $examId, $latestSubscriptionDate);;

        if ($attemptsUsed >= $exam->attempt_num) {
            $this->setError('You have exhausted your attempts');
            return false;
        }

        // Create a result record to log this attempt
        $resultRecord = $this->examRepository->createResultRecord($userId, $exam, $attemptsUsed);

        return [
            'exam' => $exam,
            'settings' => $this->prepareExamSettings($exam, $resultRecord->id),
        ];
    }

    public function getTestExam()
    {
        $exam = $this->examRepository->getExamWithCategoryAndQuestions($this->examRepository->getTestExam()->test_exam_id);

        if (!$exam) {
            $this->setError('Exam not found');
            return false;
        }

        return [
            'exam' => $exam,
            'settings' => $this->prepareExamSettings($exam),
        ];
    }

    private function prepareExamSettings($exam, $resultRecordId = null)
    {
        return [
            "id"=>$exam->id,
            "name" => $exam->examName,
            "description" => $exam->description,
            "photo" => $exam->photo,
            'questions_num' => $exam->questions_num,
            'attempt_num' => $exam->attempt_num,
            'duration_in_minutes' => $exam->duration_in_minutes,
            'exam_category_auto_move' => $exam->exam_category_auto_move,
            'result_record' => $resultRecordId,
        ];
    }
    public function getLatestSubscriptionDate($userId)
    {
        return $this->examRepository->getLatestSubscriptionDate($userId);
    }
    public function getActivePackageIds($userId)
    {
        return $this->examRepository->getActivePackageIds($userId);
    }
    public function getExamsBySubscribedPackagesWithResults($userId, $activePackageId, $latestSubscriptionDate)
    {
        return $this->examRepository->getExamsBySubscribedPackagesWithResults($userId, $activePackageId, $latestSubscriptionDate);
    }

    public function updateResultRecord($request)
    {
        $rules = [
            'result_record' => ['required', 'integer', 'exists:results,id'],
            'passed_exam' => ['required', 'in:0,1'],
            'json_score' => ['nullable'],
            'total_right_questions' => ['required', 'integer', 'min:0'],
            'total_wrong_questions' => ['required', 'integer', 'min:0'],
            'total_skiped_questions' => ['required', 'integer', 'min:0'],
            'total_not_answered_questions' => ['required', 'integer', 'min:0'],
        ];

        $validator = Validator::make($request, $rules);

        if ($validator->fails()) {
            $this->setError($validator->errors());
            return false;
        }

        return $this->examRepository->updateResultRecord($request);
    }
}
