<?php

namespace App\Http\Controllers\Api\Exams;

use App\ApiResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\ExamCategoryResource;
use App\Http\Resources\ExamsCollection;
use App\Models\Exam;
use App\Sys\Services\Exams\ExamService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ExamController extends Controller
{
    use ApiResponse;

    protected $examService;

    public function __construct(ExamService $examService)
    {
        $this->examService = $examService;
    }

    public function getSubscribedPackageExams($limit)
    {
        $userId = Auth::guard('api')->user()->id;

        // Check if the service returned false
        $data = $this->examService->getExamsForSubscribedPackages($userId, $limit);

        if (! $data) {
            return $this->apiResponse(400, 'No data', null, null);
        }
        // data here can be used separately when integrating it with the frontend
        $exams = $data['exams'];
        $currentAttempts = $data['current_attempts'];

        // Transform the collection
        $exams->getCollection()->transform(function ($exam) use ($currentAttempts) {
            $exam->attempts_count = $currentAttempts->get($exam->id, 0);

            return $exam;
        });

        $examsCollection = new ExamsCollection($exams);

        return $this->apiResponse(200, 'All data', null, $examsCollection);
    }

    public function getSubscribedPackageExamQuestions($examId)
    {
        $userId = Auth::guard('api')->user()->id;

        $data = $this->examService->getSubscribedPackageExamQuestions($userId, $examId);

        if (! $data) {
            $error = $this->examService->errors();

            return $this->apiResponse(400, 'fail', $error, null);
        }

        $exam = $data['exam'];
        $settings = $data['settings'];

        $result = [
            'exam_settings' => $settings,
            'category_questions' => ExamCategoryResource::collection($exam->examCategory),
        ];

        // exam started, Database record for the result created
        return $this->apiResponse(200, 'All data', null, $result);
    }

    public function getTestPackageExamQuestions()
    {
        $data = $this->examService->getTestExam();

        if (! $data) {
            $error = $this->examService->errors();

            return $this->apiResponse(400, 'fail', $error, null);
        }

        $exam = $data['exam'];
        $settings = $data['settings'];

        $result = [
            'exam_settings' => $settings,
            'category_questions' => ExamCategoryResource::collection($exam->examCategory),
        ];

        return $this->apiResponse(200, 'All data', null, $result);
    }

    public function saveExamResults(Request $request)
    {
        // Will update the record as it's already created when the user requested the questions - This created when the getSubscribedPackageExamQuestions() is called
        $response = $this->examService->updateResultRecord($request->all());

        if (! $response) {
            $error = $this->examService->errors();

            return $this->apiResponse(400, 'fail', $error, null);
        }

        return $this->apiResponse(200, 'All data', null, null);
    }
}
